########################################################################
#
# $Id: EasyTT.pm,v 1.10 2024/04/24 18:44:31 gosha Exp $
#
# Based on MyEaseTTEmulation.pm,v 1.27 2002/07/08 09:35:02
#
# Copyright (c) 2001-2002 Okunev Igor <gosha@prv.mts-nn.ru>
#
#      All rights reserved. This program is free software;
#      you can redistribute it and/or modify it under the
#               same terms as Perl itself.
#
########################################################################
package Template::EasyTT;

use strict;
use vars qw( @ISA @EXPORT @EXPORT_OK $VERSION $ERROR $MAX_SLICE ); 
use Fcntl qw( :flock );
use Symbol;
use Exporter;
use File::Basename;
use File::Spec;

($VERSION='$Revision: 1.10 $')=~s/^\S+\s+(\S+)\s+.*/$1/;

$MAX_SLICE	= 10000;

@ISA		= qw(Exporter);

@EXPORT		= qw( ref_to_num );

@EXPORT_OK	= qw( process filters filters_add operators_add configure );

sub new {
	my $class = shift;
	my $self = bless {}, $class;
	my $opt  = shift;
	local $_;

	$self->{CONFIG} = {
			INCLUDE_PATH	=> undef,
			SECURE_INC		=> 0,
			INCLUDE			=> 1,
			FOREACH			=> 1,
			GLOBAL_IF		=> 1,
			LOCAL_IF		=> 1,
			STOP			=> 1,
			LOCK_OK			=> 1,
			DEBUG			=> 0,
			REPLACEMENT		=> 0,
			FILTERS			=> {},
			OPERATORS		=> {},
			VARIABLES		=> {},
			PARENT			=> bless {}
		};

	$self->{FILTERS} = {
			html			=> \&f_html_quote,
			tmpl			=> \&f_tmpl_quote,
			exe				=> \&f_execute_code,
			fsize			=> \&f_fsize,
			htmlstrict		=> \&f_htmlstrict,
			duration_str	=> \&f_duration_str
		};

	$self->{OPERATORS} = {
			'<'		=> sub { ref_to_num($_[0]) <  ref_to_num($_[1]) },
			'>'		=> sub { ref_to_num($_[0]) >  ref_to_num($_[1]) },
			'<='	=> sub { ref_to_num($_[0]) <= ref_to_num($_[1]) },
			'>='	=> sub { ref_to_num($_[0]) >= ref_to_num($_[1]) },
			'=='	=> sub { ref_to_num($_[0]) == ref_to_num($_[1]) },
			'!='	=> sub { ref_to_num($_[0]) != ref_to_num($_[1]) },

			'<<'	=> sub { $_[0] << $_[1] },
			'>>'	=> sub { $_[0] >> $_[1] },

			'eq'	=> sub { $_[0] eq $_[1] },
			'ne'	=> sub { $_[0] ne $_[1] },
			'lt'	=> sub { $_[0] lt $_[1] },
			'gt'	=> sub { $_[0] gt $_[1] },
			'le'	=> sub { $_[0] le $_[1] },
			'ge'	=> sub { $_[0] ge $_[1] },

			'&'		=> sub { $_[0] & $_[1] },
			'|'		=> sub { $_[0] | $_[1] },
			'^'		=> sub { $_[0] ^ $_[1] },

			'keys'	=> sub { ref_to_num($_[0]) },
			'scalar'=> sub { ref_to_num($_[0]) },
		
			'contain' => sub { if ( index($_[0],$_[1]) > -1 ) { return 1 } else { return 0 } },
			'beginwith' => sub { if ( index($_[0],$_[1]) == 0 ) { return 1 } else { return 0 } },
			'regexp'  => sub { my $str = $_[1]; if ( $_[0] =~ /$str/ ) { return 1 } else { return 0 } }
		};

	if ( ( defined $opt ) and ( ref $opt eq 'HASH' ) ) {
		foreach ( keys %{$self->{CONFIG}} ) {
			if ( exists $opt->{$_} ) {
				$self->configure( $_, $opt->{$_} );
			}
		}
	}
	if (	( defined $self->{CONFIG}->{FILTERS} ) and 
			( ref $self->{CONFIG}->{FILTERS} eq 'HASH' ) ) {
		
		foreach ( keys %{$self->{CONFIG}->{FILTERS}} ) {
			$self->filters_add($_, $self->{CONFIG}->{FILTERS}->{$_} );
		}
	}
	if (	( defined $self->{CONFIG}->{OPERATORS} ) and 
			( ref $self->{CONFIG}->{OPERATORS} eq 'HASH' ) ) {
		
		foreach ( keys %{$self->{CONFIG}->{OPERATORS}} ) {
			$self->operators_add($_, $self->{CONFIG}->{OPERATORS}->{$_} );
		}
	}
	
	return $self;
}

sub process {
	my $self = shift;
	my ( $file, $ptr, $output_handler ) = @_;
	my ( $data, $self_char, $filter_name, $cnt, $spool, $dir, $if_bool );

	local $^W	= 0;
	#local $[	= 0;
	local $_;

	return '' unless ref $ptr eq 'HASH';

	$ERROR		= undef;

	$if_bool	= 1;

	unless ( ref $file ) {
		( $file, $dir ) = fileparse( $file );
		$data = load_src( $self, $dir, $file );
	} elsif ( ref $file eq 'SCALAR' ) {
		#$self->configure( 'INCLUDE', 0 );
		$data = $$file;
	} else {
		return '';
	}

# IF & UNLESS PARSE	( GLOBAL ONLY )

	if ( $self->configure('GLOBAL_IF') ) {
		if_parser( $self, $ptr, \$data, 1, \$if_bool );
		$self->debug( 0x01, 'GLOBAL IF', \$data, $file );
		if ( $self->configure('REPLACEMENT') == 0x01 ) {
			$self->output_data( $output_handler, \$data );
			return 1;
		}
	}

# INCLUDE PARSE

	if ( $self->configure('INCLUDE') ) {
		if ( $self->configure('SECURE_INC') ) {
			if ( $data =~ s#\[%[ \t]+INCLUDE[ \t]+(\S+)(?:[ \t]+\|[ \t]+(\S+(?:[ \t]+\|[ \t]+\S+)*?))?[ \t]+%\]\n?#
							$filter_name = $2;
							$self->filters( $filter_name,
									load_src( $self, ($self->configure('INCLUDE_PATH')||$dir), $1, 1 )
								) #xges ) {
				$if_bool = 1;
			}
		} else {
			if ( $data =~ s#\[%[ \t]+INCLUDE[ \t]+(\S+)(?:[ \t]+\|[ \t]+(\S+(?:[ \t]+\|[ \t]+\S+)*?))?[ \t]+%\]\n?#
							$filter_name = $2;
							$self->filters( $filter_name,
									load_src( $self, ($self->configure('INCLUDE_PATH')||$dir), $1 )
								) #xges ) {
				$if_bool = 1;
			}
		}
		$self->debug( 0x02, 'INCLUDE', \$data, $file );
		if ( $self->configure('REPLACEMENT') == 0x02 ) {
			$self->output_data( $output_handler, \$data );
			return 1;
		}
	}

# FOREACH PARSE

	if ( $self->configure('FOREACH') ) {
		foreach_parser( $self, $ptr, \$data );
		$self->debug( 0x04, 'FOREACH', \$data, $file );
		if ( $self->configure('REPLACEMENT') == 0x04 ) {
			$self->output_data( $output_handler, \$data );
			return 1;
		}
	}

# IF & UNLESS PARSE	

	if ( $self->configure('LOCAL_IF') ) {
		if ( $if_bool ) {
			if_parser( $self, $ptr, \$data, 0, \$if_bool );
		}
		$self->debug( 0x08, 'LOCAL_IF', \$data, $file );
		if ( $self->configure('REPLACEMENT') == 0x08 ) {
			$self->output_data( $output_handler, \$data );
			return 1;
		}
	}

# STOP PARSE

	if ( $self->configure('STOP') ) {
		$data =~ s#\[%[ \t]+STOP[ \t]+%\].*##s;
		$self->debug( 0x10, 'STOP', \$data, $file );
		if ( $self->configure('REPLACEMENT') == 0x10 ) {
			$self->output_data( $output_handler, \$data );
			return 1;
		}
	}

# ANY TAG PARSE

	any_tag_handler( $self, $ptr, \$data );

# COMMENT PARSE

	$data =~ s#\[%\\#[%#gs;

	$self->output_data( $output_handler, \$data );

	return 1;
}

sub output_data {
	my ( $self, $output_handler, $data_ref ) = @_;

	if ( ref $output_handler eq 'SCALAR' ) {
		$$output_handler = $$data_ref;
	} elsif ( ( ref $output_handler eq 'GLOB' ) or ( ref $output_handler eq 'FH' )  ) {
		print $output_handler $$data_ref;
	} elsif ( UNIVERSAL::can( $self->configure('PARENT'), 'print' ) ) {
		$self->configure('PARENT')->print( $$data_ref );
	} else {
		print $$data_ref;
	}
}

sub debug {
	my ( $self, $level, $title, $data_ref, $file_name ) = @_;

	if ( $self->configure('DEBUG') & $level ) {
		print STDERR '#'x(int(39-(length("$file_name [ $title ]")/2))), " $file_name [ $title ] ", '#'x(int(39-(length("$file_name [ $title ]")/2))), "\n";
		print STDERR "\n", $$data_ref, "\n";
		print STDERR "\n", '#'x80, "\n\n";
	}

	return 1;
}

sub ref_to_num {
	my ( $ref ) = @_;

	if ( ref $ref eq 'ARRAY' ) {
		return scalar @$ref;
	} elsif ( ref $ref eq 'HASH' ) {
		return keys %$ref;
	} elsif ( ref $ref eq 'SCALAR' ) {
		return $$ref;
	} elsif ( ref $ref eq 'CODE' ) {
		return $ref->();
	} else {
		return $ref;
	}
}

sub var_tune {
	my ( $ptr, $var ) = @_;
	my $sub_var;
	local $_;

	foreach $sub_var ( split( /\./, $var ) ) {
		if ( ref $ptr eq 'HASH' ) {
			unless ( exists $ptr->{$sub_var} ) {
				$ptr = undef;
			} else {
				$ptr = $ptr->{$sub_var}
			}
		} elsif ( ref $ptr eq 'ARRAY' ) {
			$ptr = $ptr->[$sub_var];
		} else {
			return undef;
		}
	}

	return $ptr;
}

sub any_tag_handler {
	my ( $self, $ptr, $data_ref ) = @_;
	my ( $var, $filter_name, $i, $bool, $sub_str, @chunks, %vars_ptr );
	local $_;

	$bool		= 0;
	$sub_str	= '';

	{
		@chunks = split m#\[%[ \t]+(\S+)(?:[ \t]+\|[ \t]+(\S+(?:[ \t]+\|[ \t]+\S+)*?))?[ \t]+%\]#, ( $sub_str || $$data_ref ), $MAX_SLICE;

		$$data_ref = '' unless $bool;

		unless ( ( 3 + $#chunks - $#chunks % 3 ) == ( $MAX_SLICE * 3 ) ) {
			$bool = 0;
		} else {
			if ( $chunks[ $#chunks ] eq '' ) {
				$bool = 0
			} else {
				$sub_str = $chunks[ $#chunks ];
				undef $chunks[ $#chunks ];
				$bool = 1;
			}
		}

		for ( $i = 0; $i <= $#chunks; $i += 3 ) {
			$$data_ref .= $chunks[$i];
			unless ( index( $chunks[$i+1], '.' ) > -1 )  {
				if ( length $chunks[$i+2] > 0 ) {
					$$data_ref .= $self->filters( $chunks[$i+2], $ptr->{$chunks[$i+1]} );
				} else {
					$$data_ref .= $ptr->{$chunks[$i+1]};
				}
			} else {
				if ( not exists $vars_ptr{$chunks[$i+1]} ) {
					$vars_ptr{$chunks[$i+1]} = var_tune( $ptr, $chunks[$i+1] );
				}
				if ( length $chunks[$i+2] > 0 ) {
					$$data_ref .= $self->filters( $chunks[$i+2], $vars_ptr{$chunks[$i+1]} );
				} else {
					$$data_ref .= $vars_ptr{$chunks[$i+1]};
				}
			}
		}
		redo if $bool;
	}

	return 1;
}

sub foreach_parser {
	my ( $self, $ptr, $data_ref ) = @_;
	my ( $t_data, $t_tag, $cnt, $tag, $bool, $pre_data );
	my ( @chunks, @last_tag_bool );
	local $_;

	@chunks = split /(\[%[ \t]+(?:END|FOREACH[ \t]|IF[ \t]|UNLESS[ \t]).*?[ \t]+%\]\n?)/, $$data_ref;

	$$data_ref = '';

	$cnt = $bool = 0;

	$#chunks += 2 - ( 1 + $#chunks ) % 2 if ( ( 1 + $#chunks ) % 2 );

	while ( scalar @chunks ) {
		$t_data = shift @chunks;
		$t_tag  = shift @chunks;

		( $tag ) = $t_tag =~ /(\w+)/;

		if ( $tag eq '' ) {
			$$data_ref .= $t_data;
		} elsif ( $tag eq 'FOREACH' ) {
			$pre_data .= $t_data . $t_tag;
			$bool = 1 if $cnt > 0;
			push @last_tag_bool, 1;
			$cnt++;
		} elsif ( $tag ne 'END' ) {
			if ( length $pre_data ) {
				$pre_data .= $t_data . $t_tag;
			} else {
				$$data_ref .= $t_data . $t_tag;
			}
			push @last_tag_bool, 0;
		} elsif ( pop @last_tag_bool and --$cnt == 0 ) {
			$pre_data .= $t_data;

			$pre_data =~ s#\[%[ \t]+FOREACH[ \t]+(\S+)[ \t]+=[ \t]+(\S+)[ \t]+%\]\n?(.*)#foreach_handler( $ptr, $1, $2, \$3 )#es;

			unless ( $bool ) {
				$$data_ref .= $pre_data;
			} else {
				$bool = $#chunks;
				unshift @chunks, split( /(\[%[ \t]+(?:END|FOREACH[ \t]|IF[ \t]|UNLESS[ \t]).*?[ \t]+%\]\n?)/, $pre_data );
				if ( ( 1 + $#chunks ) % 2 ) {
					splice( @chunks, $#chunks - $bool, 0, ( '' )[ 0 .. ( 1 - ( $#chunks + 1 ) % 2 ) ] );
				}
			}
			$bool		= 0;
			$pre_data	= '';
		} else {
			if ( length $pre_data ) {
				$pre_data .= $t_data . $t_tag;
			} else {
				$$data_ref .= $t_data . $t_tag;
			}
		}
	}

	$$data_ref .= $pre_data;

	return 1;
}

sub foreach_handler {
	my ( $ptr, $tmp_var, $var, $data_ref ) = @_;
	my ( @chunks, @chunks2, @tmp, $i, $ret_data, $quote_index );
	local $_;

	$ptr = var_tune( $ptr, $var );

	return '' unless ( ( ref $ptr eq 'ARRAY' ) or ( ref $ptr eq 'HASH' ) );

	$ret_data = '';

	@chunks = split /(\[%[ \t]+(?:[^'"\]]*?[ \t])?\Q$tmp_var\E(?:[ \t.].*?)*?[ \t]+%\])/, $$data_ref;

	for ( $i = 0; $i < $#chunks; $i += 2 ) {
		if ( $chunks[$i+1] =~ /(['"])/ )	{
			$quote_index = index ( $chunks[$i+1], $1, 0 );
		} else {
			$quote_index = -1;
		}
		if ( $quote_index == -1 ) {
			@tmp = split(/[ \t]\Q$tmp_var\E([ \t.])/, $chunks[$i+1], 3);
		} else {
			@tmp = split(/[ \t]\Q$tmp_var\E([ \t.])/, substr( $chunks[$i+1], 0, $quote_index ), 3);
			$tmp[$#tmp] .= substr( $chunks[$i+1], $quote_index );
		}
		if ( $#tmp == 4 ) {
			push @chunks2, [	$tmp[0] . ' ',
								$tmp[1] . $tmp[2] . ' ',
								$tmp[3] . $tmp[4] ];
		} else {
			push @chunks2, [	$tmp[0] . ' ',
								$tmp[1] . $tmp[2] ];
		}
	}

	if ( ref $ptr eq 'ARRAY' ) {
		for ( 0 .. scalar ( @$ptr ) - 1 ) {
			for ( $i = 0; $i < $#chunks; $i += 2 ) {
				$ret_data .= $chunks[$i] . join( $var . ".$_" , @{$chunks2[$i/2]} );
			}
			$ret_data .= $chunks[$#chunks];
		}
	} elsif ( ref $ptr eq 'HASH' ) {
		foreach ( sort keys %$ptr ) {
			for ( $i = 0; $i < $#chunks; $i += 2 ) {
				$ret_data .= $chunks[$i] . join( $var . ".$_" , @{$chunks2[$i/2]} );
			}
			$ret_data .= $chunks[$#chunks];
		}
	}

	return $ret_data;
}

sub if_parser {
	my ( $self, $ptr, $data_ref, $global_only, $if_bool ) = @_;
	my ( $tag, $arg, $i, $depth, $ref, $str, $tag_pref, $tag_data, $tag_suff );
	my ( @last_tag_bool, @chunks, @bool, @skip_tags, @if_chunks );
	local $_;

	push @if_chunks, [ '', '', '' ]; # 0 - data, 1 - teg, 2 - arg

	@chunks = split /\[%[ \t]+((?:END|ELSE|FOREACH[ \t]|IF[ \t]|ELSIF[ \t]|UNLESS[ \t]).*?)[ \t]+%\](\n?)/, $$data_ref;

	$$data_ref	= '';

	if ( $global_only ) {
		$$if_bool = 0;
		for ( $i = 0; $i <= $#chunks; $i += 3 ) {
			( $tag, $arg ) = split /[ \t]+/, $chunks[$i+1], 2;
			if ( ( $tag eq 'IF' ) or ( $tag eq 'UNLESS' ) ) {
				push @skip_tags, 0;
			} elsif ( $tag ne 'ELSIF' ) {
				next;
			}
			unless ( check_global( $ptr, $arg ) ) {
				$skip_tags[$#skip_tags]	= 1;
				$$if_bool					= 1;
			}
		}
	}

	@last_tag_bool	= ();
	@bool			= ();
	$str			= '';
	$depth			= 0;

	while ( scalar @chunks ) {
		$tag_pref = shift @chunks;
		$tag_data = shift @chunks;
		$tag_suff = shift @chunks;

		( $tag, $arg ) = split /[ \t]+/, $tag_data, 2;

		if ( ( $tag eq 'END' ) and ( pop @last_tag_bool ) and ( ! $skip_tags[ pop @bool ] ) ) {
			$if_chunks[ $#if_chunks ]->[0] = $str . $tag_pref;
			push @if_chunks, [ '', 'END', '' ], [ '', '' ];
			$str = '';
		} elsif ( $tag eq 'END' ) {
			$str .= $tag_pref . '[% ' . $tag_data . ' %]' . $tag_suff;
		} elsif ( $tag eq 'FOREACH' ) {
			push @last_tag_bool, 0;
			$str .= $tag_pref . '[% ' . $tag_data . ' %]' . $tag_suff;
		} elsif ( not defined $tag ) {
			$str .= $tag_pref . $tag_suff;
		} else {
			if ( ( $tag eq 'IF' ) or ( $tag eq 'UNLESS' ) ) {
				push @bool, $depth++;
				push @last_tag_bool, 1;
			}
			if ( $skip_tags[ $bool[ $#bool ] ] ) {
				$str .= $tag_pref . '[% ' . $tag_data . ' %]' . $tag_suff;
			} else {
				$if_chunks[ $#if_chunks ]->[0] = $str . $tag_pref;
				push @if_chunks, [ '', $tag, $arg ];
				$str = '';
			}
		}
	}
	push @if_chunks, [ $str ];

	@bool		= (1);
	$depth		= 0;
	undef		$str;

	while ( $ref = shift @if_chunks ) {
		if ( $ref->[1] eq 'IF' ) {
			$depth++;
			unless ( $bool[$depth-1] == 1 ) {
				$bool[$depth] = -1;
			} elsif ( if_bool( $self, $ptr, $ref->[2] ) ) {
				$bool[$depth] = 1;
				$$data_ref .= $ref->[0];
			} else {
				$bool[$depth] = 0;
			}
		} elsif ( $ref->[1] eq 'UNLESS' ) {
			$depth++;
			unless ( $bool[$depth-1] == 1 ) {
				$bool[$depth] = -1;
			} elsif ( ! if_bool( $self, $ptr, $ref->[2] ) ) {
				$bool[$depth] = 1;
				$$data_ref .= $ref->[0];
			} else {
				$bool[$depth] = 0;
			}
		} elsif ( $ref->[1] eq 'ELSIF' ) {
			if ( $bool[$depth] == 0 )  {
				if ( if_bool( $self, $ptr, $ref->[2] ) ) {
					$bool[$depth] = 1;
					$$data_ref .= $ref->[0];
				}
			}
		} elsif ( $ref->[1] eq 'ELSE' ) {
			if ( $bool[$depth] == 0 ) {
				$bool[$depth] = 1;
				$$data_ref .= $ref->[0];
			} else {
				$bool[$depth] = 0;
			}
		} elsif ( $ref->[1] eq 'END' ) {
			$bool[$depth] = -1;
			$depth--;
		} elsif ( $bool[$depth] == 1 ) {
			$$data_ref .= $ref->[0];
		}
	}

	return 1;
}

sub check_global {
	my ( $ptr, $var ) = @_;
	my ( $op, $check_var, $tmp_ptr );
	local $_;

	return 1 if $var eq '';

	( $var, $op, $check_var ) = split( /[ \t]+/, $var, 3 );

	$tmp_ptr = $ptr;

	$ptr = var_tune( $ptr, $var );

	unless ( $op eq '' ) {
		if ( defined ( $_ = is_str( $check_var ) ) ) {
			$tmp_ptr = $_;
		} else {
			$check_var =~ s#[ \t]+$##;
			$tmp_ptr = var_tune( $tmp_ptr, $check_var );
		}
	}

	if ( ( defined $ptr ) and ( defined $tmp_ptr ) ) {
		return 1;
	}
	return 0;
}

sub if_bool {
	my ( $self, $ptr, $var ) = @_;
	my ( $op, $check_var, $tmp_ptr );
	local $_;

	( $var, $op, $check_var ) = split( /[ \t]+/, $var, 3 );

	$tmp_ptr = $ptr;

	$ptr = var_tune( $ptr, $var );

	if ( defined ( $_ = is_str( $check_var ) ) ) {
		$tmp_ptr = $_;
	} else {
		$check_var =~ s#[ \t]+$##;
		$tmp_ptr = var_tune( $tmp_ptr, $check_var );
	}
	if (	( exists $self->{OPERATORS}->{$op} ) and
			( ref $self->{OPERATORS}->{$op} eq 'CODE' ) ) {

		return $self->{OPERATORS}->{$op}->( $ptr, $tmp_ptr );
	} else {
		unless ( defined $ptr ) {
			return 0;
		} elsif ( ref $ptr ) {
			return 1;
		} else {
			return 1 if $ptr;
		}
	}
}

sub is_str {
	my ( $str ) = @_;
	local $_;

	if ( $str =~ s/^\s*(['"])(.*?)\1\s*$/$2/ ) {
		$str =~ s#\\%\]#%]#g;
		$str =~ s#%([0-9a-fA-F][0-9a-fA-F])#pack('H*',$1)#eg;
		return $str;
	} else {
		return undef;
	}
}

sub configure {
	my ( $self, $opt, $opt_var ) = @_;

	if ( ( defined $opt ) and ( exists $self->{CONFIG}->{$opt} ) ) {
		if ( scalar @_ > 2 ) {
			$self->{CONFIG}->{$opt} = $opt_var;
		}
		return $self->{CONFIG}->{$opt};
	} else {
		return undef;
	}
}

sub filters {
	my ( $self, $f_name, $str ) = @_;
	my ( $local_fn );
	local $_;

	return '' unless length $str;

	foreach $local_fn ( split (/[ \t]+\|[ \t]+/, $f_name ) ) {
		if (	( defined $local_fn ) and
				( exists $self->{FILTERS}->{$local_fn} ) and
				( ref $self->{FILTERS}->{$local_fn} eq 'CODE' ) ) {

			$str = $self->{FILTERS}->{$local_fn}->($self,$str);
		}
	}

	return $str;
}

sub filters_add {
	my ( $self, $f_name, $f_handler ) = @_;

	if ( ( defined $f_name ) and ( defined $f_handler ) ) {
		$self->{FILTERS}->{$f_name} = $f_handler;
	} elsif ( wantarray ) {
		return keys %{$self->{FILTERS}};
	} else {
		return 0
	}

	return 1;
}

sub operators_add {
	my ( $self, $op_name, $op_handler ) = @_;

	if ( ( defined $op_name ) and ( defined $op_handler ) ) {
		$self->{OPERATORS}->{$op_name} = $op_handler;
	} elsif ( wantarray ) {
		return keys %{$self->{OPERATORS}};
	} else {
		return 0
	}

	return 1;
}

sub f_html_quote {
	my ( $self, $str ) = @_;

	$str =~ s#&#&amp;#gs;
	$str =~ s#"#&quot;#gs;
	$str =~ s#<#&lt;#gs;
	$str =~ s#>#&gt;#gs;
	$str =~ s#'#&apos;#gs;

	return $str;
}

sub f_tmpl_quote {
	my ( $self, $str ) = @_;

	$str =~ s#\[%#[%\\#gs;

	return $str;
}

# duration str
sub f_duration_str {
	my $duration = $_[1] || 0; 
	
	my $h = int($duration/3600);
	my $m = $duration%3600/60;
	my $s = $duration%60;
	$duration = sprintf('%2.2d:%2.2d:%2.2d', $h, $m, $s );
	
	return $duration
}
		
sub f_fsize {
	my $size = $_[1];
	
	if ( $size > 1024*1024) { return sprintf('%.2f M',($size/1024/1024)) }
	if ( $size > 1024) { return sprintf('%.2f K',($size/1024)) }
	
	return $size
}
		
sub f_htmlstrict {
	my $str = $_[1];
	
	$str =~ s#\%#\%25#gs;
	$str =~ s#\+#\%2B#gs;
	$str =~ s#;#\%3B#gs;
	$str =~ s#&#&amp;#gs;
	$str =~ s#\x22#&quot;#gs;
	$str =~ s#<#&lt;#gs;
	$str =~ s#>#&gt;#gs;
	$str =~ s#\x27#&apos;#gs;
	
	return $str;
}

sub f_execute_code {
	my ( $self, $code_ref ) = @_;

	if ( ref $code_ref eq 'CODE' ) {
		return $code_ref->();
	} else {
		return $code_ref;
	}
}

sub load_src {
	my ( $self, $dir, $file, $sec ) = @_;
	my ( $data, $fh, $delim );
	local $_;
	local $/;

	( $file ) = ( fileparse($file) )[0] if $sec;

	$file = File::Spec->catfile( $dir , $file );

	unless ( -e $file ) {
		$ERROR = "File not found $file";
		return '' 
	}

# Do not work in MSWin32 share volume ( aka '\\machine\dir\subdir...' )

	$fh = gensym;

	unless ( open( $fh, '<' . $file ) ) {
		$ERROR = "Can't open file $file: $!";
		ungensym $fh;
		return '';
	}
	if ( $self->configure('LOCK_OK') ) {
		flock( $fh, LOCK_SH );
	}
	binmode $fh;
	undef $/;	$data = <$fh>; 	$/ = "\n";
	if ( $self->configure('LOCK_OK') ) {
		flock( $fh, LOCK_UN );
	}
	unless ( close($fh) ) {
		$ERROR = "Can't close file $file: $!";
	}
	ungensym $fh;

	return $data;
}

1;

